
package easik.sketch.document;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;

import easik.Easik;


/**
 * Stores information about the current document.  This information includes the name of the sketch, the authors of 
 * sketch, a description, and creation/last modification dates.
 * <br>
 * Information about dates is created automatically, while other information is provided by the user through a
 * <b>DocInfoUI</b> object.
 * @author Vera Ranieri 2006
 * @author Kevin Green 2006
 * @since 2006-07-14 Vera Ranieri
 * @version 2006-08-02 Kevin Green
 *
 */
public class DocumentInfo {

	/**
	 * The name of this document
	 */
	private String _name;
	/**
	 * The authors of this document
	 */
	private ArrayList<String> _authors;
	/**
	 * The description of this document
	 */
	private String _desc;
	/**
	 * The creation date of this document
	 */
	private String _creationDate;
	/**
	 * The last modification date of this document
	 */
	private String _lastMod;
	
	/**
	 * Standard constructor with empty strings for all document information.
	 */ 
	public DocumentInfo(){
		_name = "";
		_authors = new ArrayList<String>();
		_desc = "";
		_creationDate = "";
		_lastMod = "";
		setDateInfo();
	}

	/**
	 * Gets the name of this sketch
	 * @return The name of this sketch
	 */
	public String getName() {
		return _name;
	}

	/**
	 * Sets the name of this sketch, and also sets the name above the information tree.
	 * @param name The name of this sketch
	 */
	public void setName(String name) {
		Easik.getInstance().getFrame().setTreeName(name);
		_name = name;
	}

	/**
	 * Gets the authors of this sketch
	 * @return An array list of the authors
	 */
	public ArrayList<String> getAuthors() {
		return _authors;
	}
	
	/**
	 * Returns a single string of the authors, with commas acting as separators
	 * 
	 * @return A single string of all authors
	 */
	public String getAuthorString(){
		if(_authors.size() != 0){
			String author  = "";
			
			for(String aut : _authors){
				author += aut + ", ";
			}
			//remove last comma and space.
			author = author.substring(0, author.length()-2);
			return author;
		}
		else 
			return "";
	}
	
	/**
	 * Adds an author to the list of authors.
	 * @param author The authors name
	 */
	public void addAuthor(String author){
		_authors.add(author);
	}

	/**
	 * Gets the description of the current sketch
	 * @return The description of the sketch
	 */
	public String getDesc() {
		return _desc;
	}

	/**
	 * Sets the description of the current sketch
	 * @param desc The description of the sketch
	 */
	public void setDesc(String desc) {
		_desc = desc;
	}

	/**
	 * Gets the creation date.  If the creation date is the empty string, it returns "N/A".
	 * @return The creation date if set, "N/A" otherwise.
	 */
	public String getCreationDate() {
		if(_creationDate.equals("")){
			return "N/A";
		}
		return _creationDate;
	}

	/**
	 * Sets the creation date.  If the <it>date</it> is the empty string, it sets creation date as the current date.
	 * 
	 * @param date The date of creation of this sketch, based on the first save.
	 */
	public void setCreationDate(String date) {
		if(date.equals(""))
			_creationDate = getDateTime();
		else
			_creationDate = date;
	}

	/**
	 * Gets the last modification date.  If the last modification date has not been set, it returns "N/A"
	 * @return The last modification date if set, "N/A" otherwise.
	 */
	public String getLastMod() {
		if(_lastMod.equals(""))
			return "N/A";
		return _lastMod;
	}
	
	/**
	 * Sets the last modification date.  If <it>date</it> is the empty string, it sets the last modifation date
	 * to the current date/time.
	 * @param date The date of the last modification, or the empty string if this is the latest modification.
	 */
	public void setModificationDate(String date) {
		if(date.equals(""))
			_lastMod = getDateTime();
		else
			_lastMod = date;
	}
	
	/**
	 * Sets all editable information, with the parameters determined by the user.  If any parameters have 
	 * been changed, the sketch is set to dirty so user will be prompted for a save if an attempt to discard
	 * the current sketch is made before a save.
	 * @param name The name of the sketch
	 * @param author The string of all authors of the sketch
	 * @param desc The description of the sketch
	 */
	public void setAllInfo(String name, String author, String desc){
		if (!_name.equals(name)|| !getAuthorString().equals(author) || !_desc.equals(desc)){
			Easik.getInstance().getFrame().getSketch().setDirty(true);
		
			setName(name);
			
			String[] auts = author.split(",");
			_authors = new ArrayList<String>();
			for(String aut : auts){
				
				aut = aut.trim();
				
				if(!aut.equals(""))
					_authors.add(aut);
			}
			
			_desc = desc;
		}
		
	}
	
	/**
	 * Sets the date information when a save is called.  This information is set by leaving the current creation date
	 * and resetting the last modification date to the current date.
	 */
	public void setDateInfo(){
		setCreationDate(_creationDate);
		setModificationDate("");
	}
	
	/**
	 * Determines the current date/time and formats it in the XML dateTime format
	 * 
	 * @return The current date/time formatted as a string for XML
	 */
	private static String getDateTime(){
		GregorianCalendar cal = new GregorianCalendar();
		cal.setTimeInMillis(System.currentTimeMillis());
		
		int year = cal.get(Calendar.YEAR);
		ArrayList<String> values = new ArrayList<String>(5);
		values.add(Integer.toString((cal.get(Calendar.MONTH)+1)));
	
		values.add(Integer.toString((cal.get(Calendar.DATE))));
		values.add(Integer.toString((cal.get(Calendar.HOUR_OF_DAY))));
		values.add(Integer.toString((cal.get(Calendar.MINUTE))));
		values.add(Integer.toString((cal.get(Calendar.SECOND))));
		
		for(int i = 0; i< values.size();i++){
			if(values.get(i).length()==1)
				values.set(i, "0"+ values.get(i));
			
		}
	
		return year +"-"+ values.get(0)+"-"+values.get(1)+"T"+values.get(2)+":"+values.get(3)+":"+values.get(4);
	}
}
